//////////////////////////////////////////////
// PastCode.h
//
// Some code from last tutorials used
// to initialize and clean a bit the main
// file
//////////////////////////////////////////////

/// Includes ---------------------------------

// nkLog
#include <NilkinsLog/Loggers/ConsoleLogger.h>

// nkGraphics
#include <NilkinsGraphics/Buffers/Buffer.h>
#include <NilkinsGraphics/Buffers/BufferManager.h>

#include <NilkinsGraphics/Cameras/Camera.h>
#include <NilkinsGraphics/Cameras/CameraManager.h>

#include <NilkinsGraphics/Encoders/Obj/ObjEncoder.h>

#include <NilkinsGraphics/Entities/Entity.h>

#include <NilkinsGraphics/Graph/Node.h>
#include <NilkinsGraphics/Graph/NodeManager.h>

#include <NilkinsGraphics/Log/LogManager.h>

#include <NilkinsGraphics/Meshes/Utils/MeshUtils.h>

#include <NilkinsGraphics/Meshes/Mesh.h>
#include <NilkinsGraphics/Meshes/MeshManager.h>

#include <NilkinsGraphics/Programs/Program.h>
#include <NilkinsGraphics/Programs/ProgramManager.h>
#include <NilkinsGraphics/Programs/ProgramSourcesHolder.h>

#include <NilkinsGraphics/RenderQueues/RenderQueue.h>
#include <NilkinsGraphics/RenderQueues/RenderQueueManager.h>

#include <NilkinsGraphics/Shaders/Memory/ConstantBuffer.h>
#include <NilkinsGraphics/Shaders/Memory/ShaderInstanceMemorySlot.h>
#include <NilkinsGraphics/Shaders/Memory/ShaderPassMemorySlot.h>

#include <NilkinsGraphics/Shaders/Shader.h>
#include <NilkinsGraphics/Shaders/ShaderManager.h>

#include <NilkinsGraphics/System.h>

// nkResources
#include <NilkinsResources/ResourceManager.h>

// nkWinUi
#include <NilkinsWinUi/System.h>

// Standards
#include <chrono>

#include <memory>

/// Internals : Buffers ----------------------

void prepareRaytracedBuffer ()
{
	nkGraphics::Buffer* buffer = nkGraphics::BufferManager::getInstance()->createOrRetrieve("raytracedBuffer") ;

	buffer->prepareForComputeResourceUsage() ;
	buffer->prepareForShaderResourceUsage() ;

	buffer->setElementByteSize(4 * sizeof(float)) ;
	buffer->setElementCount(800 * 600) ;

	buffer->load() ;
}

/// Internals : Mesh -------------------------

void prepareMeshFromFile ()
{
	// Initialize mesh to use later
	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;

	// Load data from file
	nkMemory::String absPath = nkResources::ResourceManager::getInstance()->getAbsoluteFromWorkingDir("sphere.obj") ;
	nkMemory::Buffer objData = nkResources::ResourceManager::getInstance()->loadFileIntoMemory(absPath) ;

	// We change some settings to alter the way the mesh is imported
	nkGraphics::ObjDecodeOptions objOptions ;
	objOptions._invertUvY = true ;
	objOptions._invertWindingOrder = true ;
	nkGraphics::DecodedData objDecoded = nkGraphics::ObjEncoder::decode(objData, objOptions) ;

	// Fill mesh
	nkGraphics::MeshFillOptions fillOptions ;
	fillOptions._autoLoad = true ;
	fillOptions._dataFillType = nkGraphics::DATA_FILL_TYPE::FORWARD ;
	nkGraphics::MeshUtils::fillMeshFromDecodedData(objDecoded._meshData[0], mesh, fillOptions) ;
}

void addMeshToRenderQueue ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;

	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;

	nkGraphics::Entity* ent = rq->addEntity() ;
	ent->setRenderInfo(nkGraphics::EntityRenderInfo(mesh, nullptr)) ;
}

void addEntityToGraph ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;
	nkGraphics::Entity* ent = rq->getEntity(0) ;

	nkGraphics::Node* node = nkGraphics::NodeManager::getInstance()->createOrRetrieve("node") ;
	node->setPositionAbsolute(nkMaths::Vector(0.f, 0.f, 10.f)) ;

	ent->setParentNode(node) ;
}

/// Internals : Shaders ----------------------

void prepareBufferCopyProgram ()
{
	nkGraphics::Program* bufferCopyProgram = nkGraphics::ProgramManager::getInstance()->createOrRetrieve("bufferCopyProgram") ;
	nkGraphics::ProgramSourcesHolder sources ;

	sources.setVertexMemory
	(
		R"eos(
			struct VertexInput
			{
				float4 position : POSITION ;
				float2 uvs : TEXCOORD0 ;
			} ;

			struct PixelInput
			{
				float4 position : SV_POSITION ;
				float2 uvs : TEXCOORD0 ;
				uint4 texInfos : TEXINFOS ;
			} ;

			cbuffer passConstants
			{
				uint4 texInfos ;
			}

			PixelInput main (VertexInput input)
			{
				PixelInput result ;

				result.position = input.position ;
				result.uvs = input.uvs ;
				result.texInfos = texInfos ;

				return result ;
			}
		)eos"
	) ;

	sources.setPixelMemory
	(
		R"eos(
			struct PixelInput
			{
				float4 position : SV_POSITION ;
				float2 uvs : TEXCOORD0 ;
				uint4 texInfos : TEXINFOS ;
			} ;

			StructuredBuffer<float4> inputBuf : register(t0) ;

			float4 main (PixelInput input) : SV_TARGET
			{
				uint2 index = uint2(input.uvs.x * input.texInfos.x, input.uvs.y * input.texInfos.y) ;
				return inputBuf[index.x + index.y * input.texInfos.x] ;
			}
		)eos"
	) ;

	bufferCopyProgram->setFromMemory(sources) ;
	bufferCopyProgram->load() ;
}

void prepareBufferCopyShader ()
{
	nkGraphics::Shader* bufferCopyShader = nkGraphics::ShaderManager::getInstance()->createOrRetrieve("bufferCopy") ;
	nkGraphics::Program* bufferCopyProgram = nkGraphics::ProgramManager::getInstance()->get("bufferCopyProgram") ;

	bufferCopyShader->setProgram(bufferCopyProgram) ;

	nkGraphics::ConstantBuffer* cBuffer = bufferCopyShader->addConstantBuffer(0) ;

	nkGraphics::ShaderPassMemorySlot* slot = cBuffer->addPassMemorySlot() ;
	slot->setAsTargetSize() ;

	nkGraphics::Buffer* buffer = nkGraphics::BufferManager::getInstance()->createOrRetrieve("raytracedBuffer") ;
	bufferCopyShader->addTexture(buffer, 0) ;

	bufferCopyShader->load() ;
}

/// Internals : Base -------------------------

void baseInit ()
{
	// Prepare the mesh we will show
	prepareMeshFromFile() ;
	addMeshToRenderQueue() ;
	addEntityToGraph() ;

	// Load the post processing ones
	prepareBufferCopyProgram() ;
	prepareBufferCopyShader() ;
}

void renderLoop (nkGraphics::RenderContext* context)
{
	// Get some data to prepare for the rendering loop
	nkGraphics::Camera* camera = nkGraphics::CameraManager::getInstance()->getDefaultCamera() ;

	nkGraphics::System* nkGraphicsSystem = nkGraphics::System::getInstance() ;
	nkWinUi::System* nkWinUiSystem = nkGraphicsSystem->getUiSystem() ;

	const float loopTimeMs = 5 * 1000.f ;
	std::chrono::system_clock::time_point firstNow = std::chrono::system_clock::now() ;

	const nkMaths::Vector sphereCenter = nkMaths::Vector(0.f, 0.f, 10.f) ;

	// Our rendering loop is ready to start
	while (nkGraphicsSystem->getHasRunToContinue())
	{
		// Frame the graphics system
		if (!nkGraphicsSystem->frame(context))
			break ;

		// Loop the camera around the sphere
		std::chrono::system_clock::time_point now = std::chrono::system_clock::now() ;
		float currentTimeMs = std::chrono::duration_cast<std::chrono::microseconds>(now - firstNow).count() / 1000.f ;
		
		float currentLoopT = (std::fmod(currentTimeMs, loopTimeMs) / loopTimeMs) * 2.f - 1.f ;
		currentLoopT *= 3.14 ;
		
		nkMaths::Vector loopPos (std::cos(currentLoopT) * 10.f, 0.f, std::sin(currentLoopT) * 10.f) ;
		loopPos += sphereCenter ;

		camera->setPositionAbsolute(loopPos) ;
		camera->lookAt(sphereCenter, nkMaths::Vector(0.f, 1.f, 0.f)) ;

		// Tick the windowing system
		nkWinUiSystem->tick() ;
	}
}